"use strict";
(() => {
  // src/plugin/plugin.js
  function isExportableFrame(node) {
    return node.type === "FRAME" || node.type === "SLIDE";
  }
  function computeGradientAngle(gradientTransform) {
    if (!gradientTransform)
      return 0;
    const a = gradientTransform[0][0];
    const c = gradientTransform[1][0];
    const angleDeg = Math.atan2(c, a) * (180 / Math.PI);
    return Math.round(angleDeg * 100) / 100;
  }
  var userExportScale = 1;
  var useNativeShapes = true;
  var convertTextAsNative = true;
  var collectedFonts = /* @__PURE__ */ new Set();
  function dynamicScale(node) {
    if (userExportScale) {
      const area2 = node.width * node.height;
      if (area2 > 2e6 && userExportScale > 2)
        return 2;
      return userExportScale;
    }
    const area = node.width * node.height;
    if (area < 2e5)
      return 3;
    if (area < 8e5)
      return 2;
    return 1;
  }
  function getRotation(node) {
    if ("rotation" in node)
      return node.rotation;
    if (node.relativeTransform) {
      const a = node.relativeTransform[0][0];
      const c = node.relativeTransform[1][0];
      const angle = Math.atan2(c, a);
      return angle * 180 / Math.PI;
    }
    return 0;
  }
  function getStyleForRange(textNode, start, end) {
    try {
      var result = {
        fontName: textNode.getRangeFontName(start, end),
        fontSize: textNode.getRangeFontSize(start, end),
        fills: textNode.getRangeFills(start, end),
        textDecoration: textNode.getRangeTextDecoration(start, end)
      };
      try {
        result.textCase = textNode.getRangeTextCase(start, end);
      } catch (e) {
        result.textCase = "ORIGINAL";
      }
      try {
        result.hyperlink = textNode.getRangeHyperlink(start, end);
      } catch (e) {
        result.hyperlink = null;
      }
      return result;
    } catch (err) {
      console.warn("Could not get style for range", err);
      return null;
    }
  }
  function isRangeUniform(textNode, start, end) {
    try {
      if (textNode.getRangeFontName(start, end) === figma.mixed)
        return false;
      if (textNode.getRangeFontSize(start, end) === figma.mixed)
        return false;
      if (textNode.getRangeFills(start, end) === figma.mixed)
        return false;
      if (textNode.getRangeTextDecoration(start, end) === figma.mixed)
        return false;
      try {
        if (textNode.getRangeTextCase(start, end) === figma.mixed)
          return false;
      } catch (e) {
      }
      try {
        if (textNode.getRangeHyperlink(start, end) === figma.mixed)
          return false;
      } catch (e) {
      }
      return true;
    } catch (e) {
      return false;
    }
  }
  function findRunEnd(textNode, start, totalLength) {
    let window = 1;
    let runEnd = start + 1;
    while (runEnd < totalLength) {
      window *= 2;
      const testEnd = Math.min(start + window, totalLength);
      if (!isRangeUniform(textNode, start, testEnd)) {
        let lo = runEnd, hi = testEnd;
        while (lo < hi) {
          const mid = Math.floor((lo + hi + 1) / 2);
          if (isRangeUniform(textNode, start, mid)) {
            lo = mid;
          } else {
            hi = mid - 1;
          }
        }
        return lo;
      }
      runEnd = testEnd;
    }
    return totalLength;
  }
  function getStyledText(textNode) {
    const runs = [];
    const totalLength = textNode.characters.length;
    if (totalLength === 0)
      return runs;
    let start = 0;
    while (start < totalLength) {
      const runEnd = findRunEnd(textNode, start, totalLength);
      const currentStyle = getStyleForRange(textNode, start, start + 1);
      const runText = textNode.characters.substring(start, runEnd);
      const fontNameObj = currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.fontName;
      const styleStr = fontNameObj && fontNameObj.style ? fontNameObj.style.toLowerCase() : "";
      const bold = styleStr.includes("bold");
      const italic = styleStr.includes("italic");
      const underline = (currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.textDecoration) === "UNDERLINE";
      const strikethrough = (currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.textDecoration) === "STRIKETHROUGH";
      const textCase = (currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.textCase) || "ORIGINAL";
      const hyperlinkObj = currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.hyperlink;
      const hyperlink = hyperlinkObj && hyperlinkObj.type === "URL" ? hyperlinkObj.value : null;
      let fontColor;
      let gradientStops;
      let gradientAngle = 0;
      let fillOpacity;
      const fills = currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.fills;
      if (fills !== figma.mixed && Array.isArray(fills) && fills.length > 0) {
        const first = fills.find((f) => f.visible !== false && (f.opacity === void 0 || f.opacity > 0));
        if (!first) {
        } else {
          if (typeof first.opacity === "number" && first.opacity < 1) {
            fillOpacity = first.opacity;
          }
          if (first.type === "SOLID") {
            const fill = first;
            fontColor = {
              r: Math.round(fill.color.r * 255),
              g: Math.round(fill.color.g * 255),
              b: Math.round(fill.color.b * 255)
            };
          } else if (first.type === "GRADIENT_LINEAR") {
            const grad = first;
            const stops = grad.gradientStops || [];
            if (stops.length >= 2) {
              gradientStops = stops.map((st) => ({
                position: st.position,
                color: {
                  r: Math.round(st.color.r * 255),
                  g: Math.round(st.color.g * 255),
                  b: Math.round(st.color.b * 255)
                }
              }));
              gradientAngle = computeGradientAngle(grad.gradientTransform);
            } else if (stops.length === 1) {
              fontColor = {
                r: Math.round(stops[0].color.r * 255),
                g: Math.round(stops[0].color.g * 255),
                b: Math.round(stops[0].color.b * 255)
              };
            }
          }
        }
      }
      const ls = textNode.getRangeLetterSpacing(start, runEnd);
      const letterSpacingPx = ls.unit === "PIXELS" ? ls.value : (currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.fontSize) * (ls.value / 100);
      const psObj = textNode.paragraphSpacing;
      let paragraphSpacingPx;
      if (psObj !== figma.mixed && typeof psObj === "number" && psObj > 0) {
        paragraphSpacingPx = psObj;
      }
      const lhObj = textNode.getRangeLineHeight(start, runEnd);
      let lineHeightPx;
      if (lhObj !== figma.mixed) {
        const fontSizeNum = currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.fontSize;
        if (lhObj.unit === "PIXELS") {
          lineHeightPx = lhObj.value;
        } else if (lhObj.unit === "PERCENT") {
          lineHeightPx = fontSizeNum * (lhObj.value / 100);
        }
      }
      var fontName = fontNameObj ? `${fontNameObj.family} ${fontNameObj.style}`.trim() : "Unknown";
      collectedFonts.add(fontName);
      var runObj = {
        text: runText,
        fontName,
        fontSize: currentStyle === null || currentStyle === void 0 ? void 0 : currentStyle.fontSize,
        fontColor,
        gradientStops,
        gradientAngle,
        bold,
        italic,
        underline,
        strikethrough,
        letterSpacing: letterSpacingPx,
        paragraphSpacing: paragraphSpacingPx,
        lineHeight: lineHeightPx
      };
      if (textCase !== "ORIGINAL")
        runObj.textCase = textCase;
      if (hyperlink)
        runObj.hyperlink = hyperlink;
      if (fillOpacity !== void 0)
        runObj.fillOpacity = fillOpacity;
      runs.push(runObj);
      start = runEnd;
    }
    return runs;
  }
  function shouldFlattenText(textNode) {
    if (!convertTextAsNative) {
      return { shouldFlatten: true, reason: "user disabled text conversion" };
    }
    if (Array.isArray(textNode.strokes) && textNode.strokes.length > 0) {
      return { shouldFlatten: true, reason: "has strokes" };
    }
    if (Array.isArray(textNode.effects) && textNode.effects.length > 0) {
      return { shouldFlatten: true, reason: "has effects" };
    }
    var fills = textNode.fills;
    if (Array.isArray(fills)) {
      if (fills.some(function(f) {
        return f.type === "IMAGE";
      })) {
        return { shouldFlatten: true, reason: "has image fill" };
      }
      if (fills.some(function(f) {
        return f.type === "GRADIENT_RADIAL" || f.type === "GRADIENT_ANGULAR" || f.type === "GRADIENT_DIAMOND";
      })) {
        return { shouldFlatten: true, reason: "has non-linear gradient fill" };
      }
    } else if (fills === figma.mixed) {
      var len = textNode.characters.length;
      for (var ci = 0; ci < len; ci++) {
        var charFills = textNode.getRangeFills(ci, ci + 1);
        if (Array.isArray(charFills)) {
          for (var fi = 0; fi < charFills.length; fi++) {
            var ft = charFills[fi].type;
            if (ft === "IMAGE") {
              return { shouldFlatten: true, reason: "has image fill (mixed)" };
            }
            if (ft === "GRADIENT_RADIAL" || ft === "GRADIENT_ANGULAR" || ft === "GRADIENT_DIAMOND") {
              return { shouldFlatten: true, reason: "has non-linear gradient fill (mixed)" };
            }
          }
        }
      }
    }
    if (textNode.lineHeight === figma.mixed) {
      return { shouldFlatten: true, reason: "mixed line heights" };
    }
    return { shouldFlatten: false, reason: null };
  }
  function extractTextProperties(textNode) {
    const props = {
      textContent: textNode.characters,
      textRuns: getStyledText(textNode),
      verticalAlign: textNode.textAlignVertical || "TOP",
      textAlign: textNode.textAlignHorizontal || "LEFT",
      marginTop: textNode.paddingTop || 0,
      marginBottom: textNode.paddingBottom || 0,
      marginLeft: textNode.paddingLeft || 0,
      marginRight: textNode.paddingRight || 0,
      textAutoResize: textNode.textAutoResize,
      height: textNode.height
    };
    if (textNode.paragraphIndent && textNode.paragraphIndent !== figma.mixed && textNode.paragraphIndent > 0) {
      props.paragraphIndent = textNode.paragraphIndent;
    }
    return props;
  }
  function analyzeTextNode(textNode) {
    var decision = shouldFlattenText(textNode);
    if (decision.shouldFlatten) {
      return {
        canExportNative: false,
        flattenReason: decision.reason,
        extractedData: null
      };
    }
    return {
      canExportNative: true,
      flattenReason: null,
      extractedData: extractTextProperties(textNode)
    };
  }
  function childHasEffects(child) {
    return "effects" in child && child.effects && child.effects.length > 0;
  }
  async function flattenNodeToImage(node, data, scale = 3) {
    try {
      const bytes = await node.exportAsync({
        format: "PNG",
        contentsOnly: true,
        useAbsoluteBounds: false,
        constraint: { type: "SCALE", value: scale }
      });
      data.type = isExportableFrame(node) ? "FRAME" : "EFFECT";
      data.name = node.name + " (flattened with effects)";
      data.imageData = figma.base64Encode(bytes);
      delete data.children;
      if (node.absoluteRenderBounds) {
        const absBounds = node.absoluteRenderBounds;
        data.width = absBounds.width;
        data.height = absBounds.height;
        data.x = absBounds.x;
        data.y = absBounds.y;
      }
      return data;
    } catch (err) {
      console.error("Error flattening node:", node.name, err);
      return null;
    }
  }
  var EXPORT_BATCH_SIZE = 4;
  async function processBatched(items, processFn) {
    const results = [];
    for (let i = 0; i < items.length; i += EXPORT_BATCH_SIZE) {
      const chunk = items.slice(i, i + EXPORT_BATCH_SIZE);
      const chunkResults = await Promise.all(chunk.map(processFn));
      results.push(...chunkResults);
    }
    return results;
  }
  async function handleTextNode(node, data, effectiveOpacity) {
    const textFills = node.fills;
    if (textFills !== figma.mixed && Array.isArray(textFills)) {
      const hasVisibleFill = textFills.some((f) => f.visible !== false && (f.opacity === void 0 || f.opacity > 0));
      if (!hasVisibleFill) {
        console.log("[text] Skipping invisible text: " + node.name);
        return null;
      }
    }
    const result = analyzeTextNode(node);
    if (!result.canExportNative) {
      return flattenNodeToImage(node, data, 3);
    }
    Object.assign(data, result.extractedData);
    return data;
  }
  async function handleFrameNode(node, data, effectiveOpacity) {
    const transform = node.absoluteTransform;
    const x = transform[0][2];
    const y = transform[1][2];
    const w = node.width;
    const h = node.height;
    const rot = getRotation(node);
    let backgroundImageNode = null;
    if (node.parent && isExportableFrame(node.parent) || node.effects && node.effects.length > 0) {
      backgroundImageNode = await flattenNodeToImage(node, {
        type: node.type,
        name: node.name + " (flattened)",
        x,
        y,
        width: w,
        height: h,
        rotation: rot,
        opacity: effectiveOpacity,
        flipX: false,
        flipY: false,
        textAutoResize: "NONE"
      }, Math.max(2, dynamicScale(node)));
      if (backgroundImageNode) {
        backgroundImageNode.x -= x;
        backgroundImageNode.y -= y;
      }
    }
    if (!backgroundImageNode && "fills" in node && node.fills !== figma.mixed && Array.isArray(node.fills) && node.fills.length > 0 && node.fills.some((f) => f.visible !== false)) {
      try {
        const tempRect = figma.createRectangle();
        tempRect.resize(w, h);
        tempRect.fills = node.fills;
        const pngBytes = await tempRect.exportAsync({
          format: "PNG",
          constraint: { type: "SCALE", value: Math.max(2, dynamicScale(node)) }
        });
        backgroundImageNode = {
          type: "RECTANGLE",
          name: node.name + " (background)",
          x: 0,
          y: 0,
          width: w,
          height: h,
          rotation: 0,
          opacity: 1,
          flipX: false,
          flipY: false,
          textAutoResize: "NONE",
          imageData: figma.base64Encode(pngBytes)
        };
        tempRect.remove();
      } catch (err) {
        console.error("Error exporting frame fill:", err);
      }
    }
    const visibleChildren = node.children.filter((c) => c.visible);
    const kids = (await processBatched(visibleChildren, (child) => {
      const hasStroke = "strokes" in child && child.strokeWeight > 0;
      if (child.type !== "TEXT" && (childHasEffects(child) || isExportableFrame(child) || hasStroke)) {
        const childOpacity = effectiveOpacity * ("opacity" in child ? child.opacity : 1);
        return flattenNodeToImage(child, {
          type: child.type,
          name: child.name,
          x: child.absoluteTransform[0][2],
          y: child.absoluteTransform[1][2],
          width: child.width,
          height: child.height,
          rotation: getRotation(child),
          opacity: childOpacity,
          flipX: false,
          flipY: false,
          textAutoResize: "NONE"
        }, dynamicScale(child)).catch((err) => {
          console.error("Error processing child node:", err);
          return null;
        });
      }
      return processNode(child, effectiveOpacity).catch((err) => {
        console.error("Error processing child node:", err);
        return null;
      });
    })).filter((c) => c !== null);
    for (const cd of kids) {
      cd.x -= x;
      cd.y -= y;
    }
    let filteredOutCount = 0;
    const inBoundsKids = kids.filter((cd) => {
      if (cd.width <= 0 || cd.height <= 0) {
        filteredOutCount++;
        return false;
      }
      if (cd.x + cd.width <= 0 || cd.y + cd.height <= 0 || cd.x >= w || cd.y >= h) {
        filteredOutCount++;
        return false;
      }
      return true;
    });
    if (filteredOutCount > 0) {
      console.log("[filter] " + node.name + ": removed " + filteredOutCount + " out-of-bounds/empty children");
    }
    data.children = [
      ...backgroundImageNode ? [backgroundImageNode] : [],
      ...inBoundsKids
    ];
    data.x = 0;
    data.y = 0;
    return data;
  }
  async function handleInstanceNode(node, data, effectiveOpacity) {
    try {
      const pngBytes = await node.exportAsync({
        format: "PNG",
        contentsOnly: true,
        useAbsoluteBounds: false,
        constraint: { type: "SCALE", value: dynamicScale(node) }
      });
      data.type = "RECTANGLE";
      data.name = node.name + " (flattened asset)";
      data.imageData = figma.base64Encode(pngBytes);
      delete data.children;
      if (node.absoluteRenderBounds) {
        const abs = node.absoluteRenderBounds;
        data.width = abs.width;
        data.height = abs.height;
        data.x = abs.x;
        data.y = abs.y;
      }
      return data;
    } catch (err) {
      console.error("Error flattening INSTANCE:", node.name, err);
      return null;
    }
  }
  async function handleGroupNode(node, data, effectiveOpacity) {
    try {
      const pngBytes = await node.exportAsync({
        format: "PNG",
        contentsOnly: true,
        useAbsoluteBounds: false,
        constraint: { type: "SCALE", value: dynamicScale(node) }
      });
      data.type = "GROUP";
      data.name = node.name + " (flattened)";
      data.imageData = figma.base64Encode(pngBytes);
      delete data.children;
      if (node.absoluteRenderBounds) {
        const abs = node.absoluteRenderBounds;
        data.width = abs.width;
        data.height = abs.height;
        data.x = abs.x;
        data.y = abs.y;
      }
      return data;
    } catch (err) {
      console.error("Error flattening GROUP:", node.name, err);
      return null;
    }
  }
  async function handleRectangleWithImage(node, data, effectiveOpacity) {
    const parentFrame = node.parent;
    const isClip = parentFrame != null && isExportableFrame(parentFrame) && parentFrame.clipsContent === true;
    if (isClip || node.fills[0].scaleMode !== "FILL") {
      console.log(`Flattening IMAGE node "${node.name}" due to clipContent or scaleMode`);
      return flattenNodeToImage(node, data, 3);
    } else {
      const imgBytes = await node.exportAsync({
        format: "PNG",
        contentsOnly: true,
        useAbsoluteBounds: false,
        constraint: { type: "SCALE", value: 3 }
      });
      data.imageData = figma.base64Encode(imgBytes);
      return data;
    }
  }
  async function handleFallbackNode(node, data, effectiveOpacity) {
    const rot = data.rotation;
    let origRot = null;
    if (rot !== 0 && "rotation" in node) {
      origRot = node.rotation;
      node.rotation = 0;
    }
    try {
      const pngBytes = await node.exportAsync({
        format: "PNG",
        contentsOnly: true,
        useAbsoluteBounds: true,
        constraint: { type: "SCALE", value: dynamicScale(node) }
      });
      data.imageData = figma.base64Encode(pngBytes);
    } catch (err) {
      console.error("Error exporting node:", node.name, err);
      return null;
    } finally {
      if (origRot !== null) {
        node.rotation = origRot;
      }
    }
    return data;
  }
  function isRectangleWithImage(node) {
    return node.type === "RECTANGLE" && node.fills !== figma.mixed && Array.isArray(node.fills) && node.fills.length > 0 && node.fills[0].type === "IMAGE";
  }
  function shouldConvertShapeNatively(node) {
    if (node.type === "RECTANGLE") {
      const cr = node.cornerRadius;
      if (cr === figma.mixed) {
        return { canConvertNative: false, reason: "has mixed corner radii" };
      }
      if ("topLeftRadius" in node) {
        const tl = node.topLeftRadius || 0;
        const tr = node.topRightRadius || 0;
        const bl = node.bottomLeftRadius || 0;
        const br = node.bottomRightRadius || 0;
        if (!(tl === tr && tr === bl && bl === br)) {
          return { canConvertNative: false, reason: "has non-uniform corner radii" };
        }
      }
    }
    if (Array.isArray(node.effects) && node.effects.length > 0) {
      const hasVisibleEffect = node.effects.some((e) => e.visible !== false);
      if (hasVisibleEffect) {
        return { canConvertNative: false, reason: "has effects" };
      }
    }
    const fills = node.fills;
    if (fills === figma.mixed) {
      return { canConvertNative: false, reason: "has mixed fills" };
    }
    if (Array.isArray(fills)) {
      const visibleFills = fills.filter((f) => f.visible !== false);
      if (visibleFills.length > 1) {
        return { canConvertNative: false, reason: "has multiple fills" };
      }
      if (visibleFills.length === 1) {
        const fill = visibleFills[0];
        if (fill.type !== "SOLID" && fill.type !== "GRADIENT_LINEAR") {
          return { canConvertNative: false, reason: "has unsupported fill (" + fill.type + ")" };
        }
      }
    }
    const strokes = node.strokes;
    if (strokes === figma.mixed) {
      return { canConvertNative: false, reason: "has mixed strokes" };
    }
    if (Array.isArray(strokes)) {
      const visibleStrokes = strokes.filter((s) => s.visible !== false);
      if (visibleStrokes.length > 1) {
        return { canConvertNative: false, reason: "has multiple strokes" };
      }
      if (visibleStrokes.length === 1) {
        const stroke = visibleStrokes[0];
        if (stroke.type !== "SOLID") {
          return { canConvertNative: false, reason: "has non-solid stroke (" + stroke.type + ")" };
        }
      }
    }
    return { canConvertNative: true, reason: null };
  }
  function extractShapeProperties(node) {
    const result = {
      shapeType: node.type
      // "RECTANGLE" or "ELLIPSE"
    };
    if (node.type === "RECTANGLE") {
      const cr = node.cornerRadius;
      if (cr !== figma.mixed && cr > 0) {
        result.cornerRadius = cr;
      } else if ("topLeftRadius" in node && node.topLeftRadius > 0) {
        result.cornerRadius = node.topLeftRadius;
      }
    }
    const fills = node.fills;
    if (Array.isArray(fills)) {
      const visibleFill = fills.find((f) => f.visible !== false);
      if (visibleFill) {
        if (visibleFill.type === "SOLID") {
          result.fill = {
            type: "SOLID",
            color: {
              r: Math.round(visibleFill.color.r * 255),
              g: Math.round(visibleFill.color.g * 255),
              b: Math.round(visibleFill.color.b * 255)
            },
            opacity: visibleFill.opacity !== void 0 ? visibleFill.opacity : 1
          };
        } else if (visibleFill.type === "GRADIENT_LINEAR") {
          const stops = visibleFill.gradientStops.map((stop) => ({
            position: stop.position,
            color: {
              r: Math.round(stop.color.r * 255),
              g: Math.round(stop.color.g * 255),
              b: Math.round(stop.color.b * 255)
            },
            opacity: stop.color.a !== void 0 ? stop.color.a : 1
          }));
          let angle = 0;
          if (visibleFill.gradientTransform) {
            const [[a, c], [b, d]] = visibleFill.gradientTransform;
            angle = Math.atan2(b, a) * (180 / Math.PI);
          }
          result.fill = {
            type: "GRADIENT_LINEAR",
            angle,
            stops,
            opacity: visibleFill.opacity !== void 0 ? visibleFill.opacity : 1
          };
        }
      }
    }
    const strokes = node.strokes;
    if (Array.isArray(strokes)) {
      const visibleStroke = strokes.find((s) => s.visible !== false);
      if (visibleStroke && visibleStroke.type === "SOLID") {
        result.stroke = {
          color: {
            r: Math.round(visibleStroke.color.r * 255),
            g: Math.round(visibleStroke.color.g * 255),
            b: Math.round(visibleStroke.color.b * 255)
          },
          weight: node.strokeWeight || 1,
          opacity: visibleStroke.opacity !== void 0 ? visibleStroke.opacity : 1
        };
      }
    }
    return result;
  }
  async function handleNativeShape(node, data, effectiveOpacity) {
    const props = extractShapeProperties(node);
    data.nativeShape = true;
    data.shapeType = props.shapeType;
    if (props.fill) {
      data.fill = props.fill;
    }
    if (props.stroke) {
      data.stroke = props.stroke;
    }
    return data;
  }
  var NODE_HANDLERS = {
    TEXT: handleTextNode,
    FRAME: handleFrameNode,
    SLIDE: handleFrameNode,
    INSTANCE: handleInstanceNode,
    GROUP: handleGroupNode
    // Note: RECTANGLE with IMAGE fill is handled specially before dispatch
  };
  function buildBaseData(node, parentOpacity) {
    const transform = node.absoluteTransform;
    let x = transform[0][2], y = transform[1][2];
    const w = node.width, h = node.height;
    const effectiveOpacity = parentOpacity * ("opacity" in node ? node.opacity : 1);
    const rot = getRotation(node);
    const a = transform[0][0], b = transform[0][1], c = transform[1][0], d = transform[1][1];
    const det = a * d - b * c;
    let flipX = false, flipY = false;
    if (det < 0) {
      const rotRad = rot * Math.PI / 180;
      const expectedCos = Math.cos(rotRad);
      if (Math.sign(a) !== Math.sign(expectedCos))
        flipX = true;
      if (Math.sign(d) !== Math.sign(expectedCos))
        flipY = true;
    }
    if (rot === 0) {
      if (flipX)
        x -= w;
      if (flipY)
        y -= h;
    } else {
      const x1 = x, y1 = y;
      const x2 = a * w + x, y2 = c * w + y;
      const x3 = b * h + x, y3 = d * h + y;
      const x4 = a * w + b * h + x, y4 = c * w + d * h + y;
      x = Math.min(x1, x2, x3, x4);
      y = Math.min(y1, y2, y3, y4);
    }
    const data = {
      type: node.type,
      name: node.name,
      x,
      y,
      width: w,
      height: h,
      rotation: rot,
      opacity: effectiveOpacity,
      flipX,
      flipY,
      textAutoResize: "NONE"
    };
    return { data, effectiveOpacity };
  }
  async function processNode(node, parentOpacity = 1) {
    if (!node.visible)
      return null;
    if (node.width === 0 || node.height === 0)
      return null;
    const { data, effectiveOpacity } = buildBaseData(node, parentOpacity);
    if (isRectangleWithImage(node)) {
      return handleRectangleWithImage(node, data, effectiveOpacity);
    }
    if (useNativeShapes && (node.type === "RECTANGLE" || node.type === "ELLIPSE")) {
      const analysis = shouldConvertShapeNatively(node);
      if (analysis.canConvertNative) {
        console.log("[native] Converting " + node.type + " '" + node.name + "' as native shape");
        return handleNativeShape(node, data, effectiveOpacity);
      } else {
        console.log("[native] Falling back to image for " + node.type + " '" + node.name + "': " + analysis.reason);
      }
    }
    const handler = NODE_HANDLERS[node.type] || handleFallbackNode;
    return handler(node, data, effectiveOpacity);
  }
  async function processSelectedFrames(frameIds) {
    let selectedFrames;
    if (frameIds && frameIds.length > 0) {
      const allFrames = figma.currentPage.children.filter(isExportableFrame);
      const frameMap = new Map(allFrames.map((f) => [f.id, f]));
      selectedFrames = frameIds.map((id) => frameMap.get(id)).filter((f) => f);
    }
    if (!selectedFrames || selectedFrames.length === 0) {
      selectedFrames = figma.currentPage.selection.filter(isExportableFrame);
    }
    if (selectedFrames.length === 0) {
      figma.notify("Please select at least one frame or slide for export.");
      return [];
    }
    figma.ui.postMessage({
      type: "extraction-progress",
      current: 0,
      total: selectedFrames.length,
      phase: "starting"
    });
    if (selectedFrames.length > 1) {
      figma.notify(`Extracting ${selectedFrames.length} frames...`, { timeout: 2e3 });
    }
    const slides = [];
    for (let i = 0; i < selectedFrames.length; i++) {
      figma.notify(`Extracting frame ${i + 1} of ${selectedFrames.length}...`, { timeout: 2e3 });
      const slide = await processNode(selectedFrames[i]);
      slides.push(slide);
      figma.ui.postMessage({
        type: "extraction-progress",
        current: i + 1,
        total: selectedFrames.length,
        phase: "extracting"
      });
    }
    figma.notify(`Processing complete for ${slides.length} frames`);
    return slides.filter((s) => s !== null);
  }
  figma.showUI(__html__, { width: 480, height: 660 });
  function sendSelectionInfo() {
    let frames = figma.currentPage.selection.filter(isExportableFrame);
    if (frames.length === 0) {
      frames = figma.currentPage.children.filter(isExportableFrame);
    }
    figma.ui.postMessage({
      type: "selection-info",
      frameCount: frames.length,
      frameNames: frames.map((f) => f.name).slice(0, 10),
      frameDimensions: frames.length > 0 ? { width: Math.round(frames[0].width), height: Math.round(frames[0].height) } : null
    });
  }
  async function exportThumbnails() {
    let frames = figma.currentPage.selection.filter(isExportableFrame);
    if (frames.length === 0) {
      frames = figma.currentPage.children.filter(isExportableFrame);
    }
    if (frames.length === 0) {
      figma.ui.postMessage({ type: "frames-loaded", frames: [] });
      return;
    }
    figma.ui.postMessage({ type: "thumbnails-loading", frameCount: frames.length });
    const thumbnails = await processBatched(frames, async (frame) => {
      try {
        const bytes = await frame.exportAsync({
          format: "PNG",
          constraint: { type: "WIDTH", value: 176 }
          // 2x retina for 88px thumb
        });
        return {
          id: frame.id,
          name: frame.name,
          width: Math.round(frame.width),
          height: Math.round(frame.height),
          thumbnail: figma.base64Encode(bytes)
        };
      } catch (err) {
        console.error("Error exporting thumbnail:", frame.name, err);
        return {
          id: frame.id,
          name: frame.name,
          width: Math.round(frame.width),
          height: Math.round(frame.height),
          thumbnail: null
        };
      }
    });
    figma.ui.postMessage({ type: "frames-loaded", frames: thumbnails });
  }
  sendSelectionInfo();
  exportThumbnails();
  figma.on("selectionchange", () => {
    sendSelectionInfo();
    exportThumbnails();
  });
  figma.ui.onmessage = async (msg) => {
    if (msg.type === "process-selection") {
      try {
        collectedFonts.clear();
        if (msg.exportScale)
          userExportScale = msg.exportScale;
        useNativeShapes = msg.useNativeShapes !== false;
        convertTextAsNative = msg.convertTextAsNative !== false;
        console.log("[settings] Native shapes:", useNativeShapes, "| Convert text:", convertTextAsNative);
        const slides = await processSelectedFrames(msg.frameIds);
        if (slides.length === 0) {
          figma.ui.postMessage({
            type: "process-error",
            error: "No frames found in selection. Please select at least one frame."
          });
          return;
        }
        figma.ui.postMessage({ type: "process-result", payload: { slides, fonts: Array.from(collectedFonts) } });
      } catch (err) {
        figma.ui.postMessage({
          type: "process-error",
          error: "Processing failed: " + err.message
        });
      }
    } else if (msg.type === "process-streaming") {
      const FRAME_TIMEOUT_MS = 45e3;
      collectedFonts.clear();
      if (msg.exportScale)
        userExportScale = msg.exportScale;
      useNativeShapes = msg.useNativeShapes !== false;
      convertTextAsNative = msg.convertTextAsNative !== false;
      console.log("[settings] Native shapes:", useNativeShapes, "| Convert text:", convertTextAsNative);
      try {
        let selectedFrames;
        if (msg.frameIds && msg.frameIds.length > 0) {
          const allFrames = figma.currentPage.children.filter(isExportableFrame);
          const frameMap = new Map(allFrames.map((f) => [f.id, f]));
          selectedFrames = msg.frameIds.map((id) => frameMap.get(id)).filter((f) => f);
        }
        if (!selectedFrames || selectedFrames.length === 0) {
          selectedFrames = figma.currentPage.selection.filter(isExportableFrame);
        }
        if (selectedFrames.length === 0) {
          figma.ui.postMessage({
            type: "process-error",
            error: "No frames or slides found in selection. Please select at least one."
          });
          return;
        }
        if (msg.maxFrames && msg.maxFrames < selectedFrames.length) {
          selectedFrames = selectedFrames.slice(0, msg.maxFrames);
        }
        const total = selectedFrames.length;
        const skipped = [];
        figma.ui.postMessage({ type: "streaming-start", total });
        for (let i = 0; i < total; i++) {
          const frameName = selectedFrames[i].name;
          console.log("[streaming] Processing frame " + (i + 1) + "/" + total + ": " + frameName);
          figma.ui.postMessage({ type: "extraction-progress", current: i + 1, total, phase: "extracting" });
          let slide = null;
          try {
            slide = await Promise.race([
              processNode(selectedFrames[i]),
              new Promise(
                (_, reject) => setTimeout(() => reject(new Error("TIMEOUT")), FRAME_TIMEOUT_MS)
              )
            ]);
          } catch (frameErr) {
            if (frameErr.message === "TIMEOUT") {
              console.error("[streaming] Frame " + (i + 1) + " timed out (45s): " + frameName);
              skipped.push({ index: i, name: frameName, reason: "timeout" });
            } else {
              console.error("[streaming] Frame " + (i + 1) + " error: " + frameName, frameErr);
              skipped.push({ index: i, name: frameName, reason: frameErr.message });
            }
            figma.ui.postMessage({ type: "slide-skipped", index: i, total, frameName, reason: frameErr.message === "TIMEOUT" ? "timeout (45s)" : frameErr.message });
            continue;
          }
          if (slide) {
            console.log("[streaming] Frame " + (i + 1) + " ready: " + frameName);
            figma.ui.postMessage({ type: "slide-ready", index: i, slide, total });
          } else {
            console.warn("[streaming] Frame " + (i + 1) + " returned null: " + frameName);
            skipped.push({ index: i, name: frameName, reason: "processNode returned null" });
            figma.ui.postMessage({ type: "slide-skipped", index: i, total, frameName, reason: "processing failed" });
          }
        }
        if (skipped.length > 0) {
          console.warn("[streaming] Skipped " + skipped.length + " frames:", skipped.map((s) => s.name).join(", "));
        }
        figma.ui.postMessage({ type: "streaming-complete", total, skippedCount: skipped.length, fonts: Array.from(collectedFonts) });
      } catch (err) {
        figma.ui.postMessage({
          type: "process-error",
          error: "Processing failed: " + err.message
        });
      }
    } else if (msg.type === "get-license") {
      try {
        const data = await figma.clientStorage.getAsync("license");
        figma.ui.postMessage({ type: "license-data", data: data || null });
      } catch (err) {
        console.error("Failed to read license from storage:", err);
        figma.ui.postMessage({ type: "license-data", data: null });
      }
    } else if (msg.type === "save-license") {
      try {
        await figma.clientStorage.setAsync("license", msg.data);
        figma.ui.postMessage({ type: "license-saved", success: true });
      } catch (err) {
        console.error("Failed to save license to storage:", err);
        figma.ui.postMessage({ type: "license-saved", success: false });
      }
    } else if (msg.type === "export-references") {
      const frameIds = msg.frameIds || [];
      if (frameIds.length === 0)
        return;
      const allFrames = figma.currentPage.children.filter(isExportableFrame);
      const frameMap = new Map(allFrames.map((f) => [f.id, f]));
      const references = [];
      for (let i = 0; i < frameIds.length; i++) {
        const frame = frameMap.get(frameIds[i]);
        if (!frame)
          continue;
        try {
          const pngBytes = await frame.exportAsync({
            format: "PNG",
            constraint: { type: "SCALE", value: 1 }
          });
          references.push({ index: i, referenceImage: figma.base64Encode(pngBytes) });
        } catch (err) {
          console.error("Failed to export reference PNG for", frame.name, err);
        }
      }
      figma.ui.postMessage({ type: "reference-images", references });
    }
  };
})();
